# Copyright 2024 warehauser @ github.com

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     https://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# views.py

import json

from datetime import datetime, timedelta
from typing import Any, List

from django.conf import settings
from django.contrib import messages
from django.contrib.auth import login, authenticate, update_session_auth_hash
from django.contrib.auth import logout as auth_logout
from django.contrib.auth.decorators import login_required
from django.contrib.auth.hashers import make_password, check_password
from django.contrib.auth.models import User
from django.core.exceptions import ObjectDoesNotExist
from django.http import HttpResponse, JsonResponse
from django.shortcuts import render, redirect, get_object_or_404
# from django.templatetags.static import static
from django.utils import timezone
from django.utils.safestring import mark_safe
from django.utils.translation import gettext as _
from django.utils import translation

from django_filters.rest_framework import DjangoFilterBackend

from rest_framework import generics, viewsets, status
from rest_framework.decorators import action
from rest_framework.exceptions import ValidationError
from rest_framework.filters import SearchFilter
from rest_framework.permissions import IsAuthenticated, AllowAny
from rest_framework.response import Response
from rest_framework.renderers import JSONRenderer

from .filters import *
from .forms import *
from .models import *
from .permissions import *
from .serializers import *

# Create your views here.

# from django.forms.boundfield import BoundField

# def apply_patches():
#     print('Applying patches')
#     if not getattr(BoundField.css_classes, 'patched', False):
#         unpatched = BoundField.css_classes

#         def css_classes(self, extra_classes=None):
#             return unpatched(self, ['form-field'] + (extra_classes or []))

#         BoundField.css_classes = css_classes
#         BoundField.css_classes.patched = True


# Authentication views

# Model view sets

class WarehauserBaseViewSet(viewsets.ModelViewSet):
    lookup_field = 'id'
    permission_classes = [WarehauserPermission,]
    renderer_classes = [JSONRenderer,]
    search_fields = ['id', 'external_id', 'options__values__contains', 'value', 'descr', 'owner']

    # API filtering
    filter_backends = [DjangoFilterBackend, SearchFilter,]

    def _protect_fields(self, user, data:list, create:bool=False):
        # Prevent altering id, updated_at, or created_at fields

        field_names = ['id', 'updated_at', 'created_at']

        if not user.is_staff and not user.is_superuser:
            field_names.append('owner')
        elif not create:
            field_names.append('owner')

        for field in field_names:
            if field in data:
                raise ValidationError(
                    {'error': _(f'Cannot set autogenerated field \'{field}\'.')}
                )

    def get_permissions(self):
        """
        Instantiates and returns the list of permissions that this view requires.
        """
        return [permission() for permission in self.permission_classes]

    def get_queryset(self):
        user = self.request.user

        # If the user is staff or superuser, they can see all objects
        if user.is_staff or user.is_superuser:
            return self.serializer_class.Meta.model.objects.all().order_by('created_at', 'updated_at')

        # Otherwise, only show objects the user has access to
        return self.serializer_class.Meta.model.objects.filter(owner__group__in=user.groups.all()).order_by('created_at', 'updated_at')

    def create(self, request, *args, **kwargs):
        user = request.user
        data = request.data

        self._protect_fields(user=user, data=data, create=True)

        if not user.is_superuser and not user.is_staff:
            owner = Client.objects.get(group__in=user.groups.all())
            if not owner:
                raise ValidationError({'error': _('User is not a member of any client group.')})

            # Set the owner field to the retrieved group
            data['owner'] = owner.id

        # return super().create(request, *args, **kwargs)
        serializer = self.get_serializer(data=data)
        serializer.is_valid(raise_exception=True)
        self.perform_create(serializer)
        return Response(serializer.data, status=status.HTTP_201_CREATED)

    def perform_create(self, serializer):
        serializer.save()

    def partial_update(self, request, *args, **kwargs):
        instance = self.get_object()
        data = request.data

        self._protect_fields(user=request.user, data=data)

        # Check for field changes
        has_changed = False
        for key, value in data.items():
            attr = getattr(instance, key, None)
            if isinstance(attr, dict) and isinstance(value, dict):
                # Update the JSONField attribute
                json = getattr(instance, key, {})
                json.update(value)
                setattr(instance, key, json)
                has_changed = True
            elif attr != value:
                # Update regular field
                setattr(instance, key, value)
                has_changed = True

        if has_changed:
            instance.updated_at = timezone.now()
            instance.save()

            return Response(
                {'message': _(f'[Update]: {instance.__class__.__name__} {instance.id} updated.')},
                status=status.HTTP_200_OK
            )

        return Response(
            {'message': _(f'[Update]: {instance.__class__.__name__} {instance.id} no change.')},
            status=status.HTTP_200_OK
        )

    def update(self, request, *args, **kwargs):
        return self.partial_update(request, *args, **kwargs)

    def destroy(self, request, *args, **kwargs):
        try:
            instance = self.get_object()
            instance.delete()
        except ObjectDoesNotExist as e:
            return Response(data={'message': e.detail}, status=e.code)
        return Response(status=status.HTTP_204_NO_CONTENT)

class WarehauserDefinitionViewSet(WarehauserBaseViewSet):
    instance_serializer_class = None

    def _do_spawn(self, request, *args, **kwargs):
        dfn = self.get_object()
        data = request.data

        instance = dfn.create_instance(data)

        return instance

    @action(detail=True, methods=['post'])
    def do_spawn(self, request, *args, **kwargs):
        instance = self._do_spawn(request=request)

        serializer = self.instance_serializer_class(instance)
        return Response(serializer.data, status=status.HTTP_200_OK)

class WarehauserInstanceViewSet(WarehauserBaseViewSet):
    def create(self, request, *args, **kwargs):
        return Response(status=status.HTTP_501_NOT_IMPLEMENTED)


# WAREHAUSE viewsets

class WarehauseDefViewSet(WarehauserDefinitionViewSet):
    instance_serializer_class = WarehauseSerializer
    serializer_class = WarehauseDefSerializer
    filterset_class = WarehauseDefFilter

class WarehauseViewSet(WarehauserInstanceViewSet):
    serializer_class = WarehauseSerializer
    filterset_class = WarehauseFilter


# PRODUCT viewsets

class ProductDefViewSet(WarehauserDefinitionViewSet):
    instance_serializer_class = ProductSerializer
    serializer_class = ProductDefSerializer
    filterset_class = ProductDefFilter

    @action(detail=True, methods=['get'], url_path='warehauses')
    def get_warehauses(self, request, id=None):
        product_def = self.get_object()
        warehauses = product_def.warehauses.all()
        serializer = WarehauseSerializer(warehauses, many=True)
        return Response(serializer.data)

class ProductViewSet(WarehauserInstanceViewSet):
    serializer_class = ProductSerializer
    filterset_class = ProductFilter


# EVENT viewsets

class EventDefViewSet(WarehauserDefinitionViewSet):
    instance_serializer_class = EventSerializer
    serializer_class = EventDefSerializer
    filterset_class = EventDefFilter

    @action(detail=True, methods=['post'])
    def do_spawn(self, request, *args, **kwargs):
        instance = super()._do_spawn(request=request)

        if not instance.is_batched:
            # process immediately
            instance.process()

        serializer = self.instance_serializer_class(instance)
        return Response(serializer.data, status=status.HTTP_200_OK)

class EventViewSet(WarehauserInstanceViewSet):
    serializer_class = EventSerializer
    filterset_class = EventFilter
